<?php

namespace Ecomerciar\CruzDelSur\ShippingMethod;

use Ecomerciar\CruzDelSur\Helper\Helper;
use Ecomerciar\CruzDelSur\Sdk\CruzDelSurSdk;

defined('ABSPATH') || class_exists('\WC_Shipping_method') || exit;

/**
 * Our main payment method class
 */
class WC_CruzDelSur extends \WC_Shipping_method
{
    /**
     * Default constructor, loads settings and MercadoPago's SDK
     */
    public function __construct($instance_id = 0)
    {
        $this->instance_id = absint($instance_id);
        // Load the settings.
        $this->init_settings();
        // Setup general properties.
        $this->setup_properties();
		$this->init();
    }

    /**
     * Establishes default settings, and loads IPN Processor
     *
     * @return void
     */
    private function setup_properties()
    {
        $this->id = 'cruzdelsur';
        $this->method_title = 'Cruz del Sur';
        $this->method_description = __('Permite a tus clientes ', 'wc-cruzdelsur');
        $this->description = $this->get_option('description');
        $this->title = $this->get_option('title');
        $this->supports = array(
            'shipping-zones',
            'instance-settings',
            'instance-settings-modal'
        );
    }

    /**
     * Declares our instance configuration
     *
     * @return void
     */

	/**
	 * Init user set variables.
	 */
	public function init() {
		$this->instance_form_fields = include __DIR__ . '/includes/settings-cruz-del-sur.php';
		$this->title                = $this->get_option( 'title' );
		$this->free_shipping        = $this->get_option( 'free_shipping' );
		$this->free_shipping_from   = $this->get_option( 'free_shipping_from' );
        $this->shipping_promise = $this->get_option('shipping_promise');
        $this->shipping_promise_day = $this->get_option('shipping_promise_day');
        $this->flat_price           =  $this->get_option( 'flat_price' ); 
        $this->flat_shipping_price  =  $this->get_option( 'flat_shipping_price' ); 

        $this->shipping_charge  =  $this->get_option( 'shipping_charge' ); 
        $this->shipping_charge_adj_type  =  $this->get_option( 'shipping_charge_adj_type' );
        $this->shipping_charge_adj_pct  =  $this->get_option( 'shipping_charge_adj_pct' ); 

        add_action( 'admin_footer', ['Ecomerciar\CruzDelSur\ShippingMethod\WC_CruzDelSur', 'enqueue_admin_js'], 10 );
	}

    /**
     * Verifica si aplica envío gratuito según el monto del carrito
     *
     * @param float $cart_total Total del carrito
     * @return bool
     */
    private function applies_free_shipping($cart_total)
    {
        return $this->free_shipping === 'yes' && $this->free_shipping_from <= $cart_total;
    }

    /**
     * Calcula el costo de envío según las reglas de negocio
     *
     * @param array $quote Cotización de la API
     * @param float $cart_total Total del carrito
     * @return float
     */
    private function calculate_shipping_cost($quote, $cart_total)
    {
        // Prioridad 1: Envío gratuito
        if ($this->applies_free_shipping($cart_total)) {
            return 0;
        }

        // Prioridad 2: Tarifa dinámica (descuento o recargo)
        if ($this->shipping_charge === 'yes') {
            $adjustment = ($this->shipping_charge_adj_type === 'DISCOUNT')
                ? (100 - floatval($this->shipping_charge_adj_pct)) / 100
                : (100 + floatval($this->shipping_charge_adj_pct)) / 100;
            return floatval($quote['price']) * $adjustment;
        }

        // Prioridad 3: Tarifa plana
        if ($this->flat_price === 'yes') {
            return floatval($this->flat_shipping_price);
        }

        // Por defecto: Precio de la API
        return floatval($quote['price']);
    }

    /**
     * Genera la etiqueta del método de envío
     *
     * @param array $quote Cotización de la API
     * @param float $cart_total Total del carrito
     * @return string
     */
    private function get_shipping_label($quote, $cart_total)
    {
        if ($this->applies_free_shipping($cart_total)) {
            return 'Cruz del Sur – Envío gratuito: ' . $quote['full_name'];
        }
        return 'Cruz del Sur – ' . $quote['full_name'];
    }

    /**
     * Calculates the shipping
     *
     * @return void
     */
    public function calculate_shipping($package = [])
    {
        // Configuración de promesa de envío
        $shipping_promise_day = 0;
        $show_shipping_promise = ($this->shipping_promise == "yes");
        if ($show_shipping_promise) {
            $shipping_promise_day = (!empty($this->shipping_promise_day)) ? $this->shipping_promise_day : 0;
        }

        $customer = Helper::get_customer_from_cart(WC()->customer);
        $items = Helper::get_items_from_cart(WC()->cart);

        // Calcular total del carrito
        $cart_total = WC()->cart->get_displayed_subtotal();
        if (WC()->cart->display_prices_including_tax()) {
            $cart_total = $cart_total - WC()->cart->get_discount_tax();
        }
        $cart_total = round($cart_total, wc_get_price_decimals());

        if (empty($items)) {
            return;
        }

        // Obtener cotizaciones
        $CdsSdk = new CruzDelSurSdk();
        $prices = $CdsSdk->get_price($customer, $items, $show_shipping_promise, $shipping_promise_day);
        
        if (!empty($prices)) {
            foreach ($prices['quotes'] as $index => $quote) {
                $this->add_rate([
                    'id'        => $this->get_rate_id() . '_' . $index,
                    'label'     => $this->get_shipping_label($quote, $cart_total),
                    'cost'      => $this->calculate_shipping_cost($quote, $cart_total),
                    'meta_data' => [
                        'quote_number' => $prices['quote_number'],
                        'line_number' => $quote['id']
                    ],
                ]);
            }
        }
    }



    public static function enqueue_admin_js() {
		wc_enqueue_js(
			"jQuery( function( $ ) {

                function wcPickitFree( el ) {
                    var form = $( el ).closest( 'form' );
					var charge_adj_type = $( '#woocommerce_cruzdelsur_shipping_charge_adj_type', form ).closest( 'tr' );
                    var charge_adj_pct = $( '#woocommerce_cruzdelsur_shipping_charge_adj_pct', form ).closest( 'tr' );          
                    if ( $( el ).prop('checked') ) {
                        charge_adj_type.show();    
                        charge_adj_pct.show();          
                    } else {         
                        charge_adj_type.hide();
                        charge_adj_pct.hide();
                    }                  
                  }            
                  $( document.body ).on( 'change', '#woocommerce_cruzdelsur_shipping_charge', function() {
                    wcPickitFree( this );
                  }); 

                  $( '#woocommerce_cruzdelsur_shipping_charge' ).trigger( 'change' );

                  $( document.body ).on( 'wc_backbone_modal_loaded', function( evt, target ) {
                    if ( 'wc-modal-shipping-method-settings' === target ) {
                        wcPickitFree( $( '#wc-backbone-modal-dialog #woocommerce_cruzdelsur_shipping_charge', evt.currentTarget ) );
                    }
                  } );

			});"
		);
	}

}
