<?php

namespace Ecomerciar\CruzDelSur\Sdk;

use Ecomerciar\CruzDelSur\Api\CruzDelSurApi;
use Ecomerciar\CruzDelSur\Helper\Helper;

class CruzDelSurSdk
{
    private $api;
    public function __construct()
    {

		if (Helper::is_sandbox()){
            $this->api = new CruzDelSurApi(
                Helper::get_option('clientid_sandbox', ''),
                Helper::get_option('username_sandbox', ''),
                Helper::get_option('password_sandbox', '')
            );
            
		} else {
            $this->api = new CruzDelSurApi(
                Helper::get_option('clientid', ''),
                Helper::get_option('username', ''),
                Helper::get_option('password', '')
            );			
		}
    }

    /**
     * Gets a quote for an order
     *
     * @param array $customer
     * @param array $items
     * @return array|false
     */
    public function get_price(array $customer, array $items, bool $show_shipping_promise = false , int $days_added = 0)
    {
        $hrs_added = $days_added * 24;
        $items_totals = [
            'volume' => 0,
            'weight' => 0,
            'price' => 0
        ];
        foreach ($items as $item) {
            $items_totals['volume'] += ($item['width'] * $item['height'] * $item['length']);
            $items_totals['weight'] += $item['weight'];
            $items_totals['price'] += $item['price'];
        }
        $weight = (string) $items_totals['weight'];
        $weight = str_replace('.', ',', $weight);
        //$volume = (string) number_format($items_totals['volume'], 2, ',', '');
        $volume = round($items_totals['volume']);
        $data_to_send = [
            'volumen' => $volume,
            'peso' => $weight,
            'codigopostal' => $customer['cp'],
            'localidad' => $customer['locality'],
            'valor' => $items_totals['price'],
            'contrareembolso' => 0,
        ];
        $res = $this->api->get('/NuevaCotXVol', $data_to_send);
        if (Helper::get_option('debug')) {
            Helper::log_debug(__FUNCTION__ . ' - Data enviada a Cruz del Sur: ' . json_encode($data_to_send));
            Helper::log_debug(__FUNCTION__ . ' - Data recibida de Cruz del Sur: ' . json_encode($res));
        }
        if ($res['Respuesta'][0]['Estado'] !== 0) {
            return false;
        }
        $data = [
            'quote_number' => $res['Respuesta'][0]['NumeroCotizacion'],
            'quotes' => []
        ];
        foreach ($res['Cotizaciones'] as $quote) {
            $newQuote = [
                'id' => $quote['CodigoLinea'],
                'name' => (empty($quote['DescripcionLarga']) ? $quote['Descripcion'] : $quote['DescripcionLarga']),
                'price' => $quote['Valor'],
                'from' => $quote['HorasDesde'] + $hrs_added,
                'to' => $quote['HorasHasta'] + $hrs_added
            ];
            $newQuote['full_name'] = $newQuote['name'] ;
            if ( $show_shipping_promise ){
              $newQuote['full_name'] = $newQuote['full_name'] . ' (' . $newQuote['from'] . '-' . $newQuote['to'] . ' Hrs)';
            }
            $data['quotes'][] = $newQuote;
        }
        return $data;
    }


    /**
     * Process an order in Cruz del Sur's Api
     *
     * @return array|false
     */
    public function process_order(\WC_Order $order, array $customer)
    {
        $shipping_methods = $order->get_shipping_methods();
        $shipping_method = array_shift($shipping_methods);
        $line_number = $shipping_method->get_meta('line_number');

        $data_to_send = [
            'idlinea' => $line_number,
            'documento' => '',
            'nombre' => $customer['full_name'],
            'telefono' => $customer['phone'],
            'email' => $customer['email'],
            //'domicilio' => $customer['full_address'],
            'domicilio' => $customer['simple_address'],
            'bultos' => 1,
            'referencia' => $order->get_id()
        ];
        $res = $this->api->get('/PedirDespachoCotizacion', $data_to_send);
        if (Helper::get_option('debug')) {
            Helper::log_debug(__FUNCTION__ . ' - Data enviada a Cruz del Sur: ' . json_encode($data_to_send));
            Helper::log_debug(__FUNCTION__ . ' - Data recibida de Cruz del Sur: ' . json_encode($res));
        }
        if ($res['Respuesta'][0]['Estado'] !== 0 && empty($res['Respuesta'][0]['NIC'])) {
            Helper::log_error('No se pudo procesar el pedido. Razón: ' . $res['Respuesta'][0]['Descripcion']);
            return false;
        }
        return $res['Respuesta'][0];
    }

    /**
     * Gets shipping label in pdf format
     *
     * @param string $order_id
     * @return string
     */
    public function get_shipping_label_pdf(string $order_id)
    {
        $sheet_size = Helper::get_option('pagesize');
        return $this->api->get_url('/EtiquetasPDF/' . $order_id, ['tamanioHoja' => $sheet_size, 'posicionArrancar' => '']);
    }

    /**
     * Gets shipping label in zpl format
     *
     * @param string $order_id
     * @return string
     */
    public function get_shipping_label_zpl(string $order_id)
    {
        return $this->api->get_url('/EtiquetasZPL/' . $order_id, ['paraReemplazarCantidades' => '']);
    }

    /**
     * Tracks an order
     *
     * @param string $order_id
     * @return array|false
     */
    public function track_order(string $nic)
    {
        $data_to_send = ['nic' => $nic];
        $res = $this->api->get('/TrackingDeUnNIC', $data_to_send);
        if (Helper::get_option('debug')) {
            Helper::log_debug(__FUNCTION__ . ' - Data enviada a Cruz del Sur: ' . json_encode($data_to_send));
            Helper::log_debug(__FUNCTION__ . ' - Data recibida de Cruz del Sur: ' . json_encode($res));
        }
        if (!empty($res['Tracking'])) {
            return $res['Tracking'];
        } else {
            return (empty($res['Respuesta'][0]['Descripcion']) ? 'Hubo un error al rastrear el pedido' : $res['Respuesta'][0]['Descripcion']);
        }
    }

    /**
     * Validates credentials from options
     * 
     * @return bool|string
     */
    public function validate_credentials()
    {

		if (Helper::is_sandbox()){
            $data_to_send = [
                'idcliente' => Helper::get_option('clientid_sandbox', '')  . '_25',
                'ulogin' =>  Helper::get_option('username_sandbox', ''),
                'uclave' => Helper::get_option('password_sandbox', '')
            ];
		} else {
            $data_to_send = [
                'idcliente' => Helper::get_option('clientid', '')  . '_25',
                'ulogin' =>  Helper::get_option('username', ''),
                'uclave' => Helper::get_option('password', '')
            ];		
		}

        $res = $this->api->get('/Prueba', $data_to_send);  
        if (Helper::get_option('debug')) {
            Helper::log_debug(__FUNCTION__ . ' - Data enviada a Cruz del Sur: ' . json_encode($data_to_send));
            Helper::log_debug(__FUNCTION__ . ' - Data recibida de Cruz del Sur: ' . json_encode($res));
        }
        if (isset($res['Respuesta']) && isset($res['Respuesta'][0]) && isset($res['Respuesta'][0]['Estado'])){
            if($res['Respuesta'][0]['Estado'] === 0){
                return true;
            } else {
                return $res['Respuesta'][0]['Descripcion'];
            }
        }
        return false;
    }
}
