<?php
namespace Ecomerciar\CruzDelSur\Lib;

class RestClient
{
    const API_BASE_URL_PROD = 'http://api-ventaenlinea.cruzdelsur.com/api/';
    const API_BASE_URL_TEST = 'http://api-test.cruzdelsur.com/api/';
    const XML_ENV_YESNO = 'carriers/cruzdelsur/ambiente';
    const XML_ENV_PROD_VALUE = 'carriers/cruzdelsur/urlenvprod';
    const XML_ENV_TEST_VALUE = 'carriers/cruzdelsur/urlenvtest';

    /**
     * @var \Ecomerciar\CruzDelSur\Helper\Helper
     */
    protected $_helper;
     /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    public function __construct(
        \Ecomerciar\CruzDelSur\Helper\Helper $helper,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
    ) {
        $this->_helper = $helper;
        $this->scopeConfig = $scopeConfig;
    }

    /**
     * Executes a HTTP Request
     *
     * @param string $method
     * @param string $endpoint
     * @param array $body
     * @param array $headers
     * @param string $env
     * @return array
     */
    public function exec($method, $endpoint, $body, $headers, $env = 'TEST')
    {
        if (!extension_loaded('curl')) {
            return $response = [
                "status" => 1000,
                "response" => 'cURL extension not found. You need to enable cURL in your php.ini or another configuration you have.'
            ];
        }

        $storeScope = \Magento\Store\Model\ScopeInterface::SCOPE_STORE;
        $getSetEnv = $this->scopeConfig->getValue(self::XML_ENV_YESNO, $storeScope);

        $urlEnvProd = $this->scopeConfig->getValue(self::XML_ENV_PROD_VALUE, $storeScope);
        $urlEnvTest = $this->scopeConfig->getValue(self::XML_ENV_TEST_VALUE, $storeScope);

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, ($getSetEnv === '1' ? $urlEnvProd : $urlEnvTest) . $endpoint);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        $api_result = curl_exec($ch);
        $api_http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($api_result === false) {
            $response = [
                "status" => 900,
                "response" => curl_error($ch)
            ];
        } else {
            $response = [
                "status" => $api_http_code,
                "response" => json_decode($api_result, true)
            ];
        }
        curl_close($ch);

        return $response;
    }

    /**
     * Makes a GET call
     *
     * @param string $endpoint
     * @param array $content
     * @param array $headers
     * @return array
     */
    public function get($endpoint, $content = [], $headers = [])
    {
        if (!empty($content))
            $endpoint .= '?' . http_build_query($content);
        return $this->exec('GET', $endpoint, [], $headers);
    }

    /**
     * Makes a POST call
     *
     * @param string $endpoint
     * @param array $content
     * @param array $headers
     * @return array
     */
    public  function post($endpoint, $content = [], $headers = [])
    {
        $headers[] = ['Content-Type' => 'Application/Json'];
        $content = json_encode($content);
        return $this->exec('POST', $endpoint, $content, $headers);
    }

    /**
     * Makes a PUT call
     *
     * @param string $endpoint
     * @param array $content
     * @param array $headers
     * @return array
     */
    public  function put($endpoint, $content = [], $headers = [])
    {
        $headers[] = ['Content-Type' => 'Application/Json'];
        $content = json_encode($content);
        return $this->exec('PUT', $endpoint, $content, $headers);
    }

    /**
     * Makes a DELETE call
     *
     * @param string $endpoint
     * @param array $content
     * @param array $headers
     * @return array
     */
    public  function delete($endpoint, $content = [], $headers = [])
    {
        $headers[] = ['Content-Type' => 'Application/Json'];
        $content = json_encode($content);
        return $this->exec('DELETE', $endpoint, $content, $headers);
    }

    public  function get_url($endpoint, $params, $env = 'TEST')
    {
        if (!empty($params))
            $endpoint .= '?' . http_build_query($params);
        $return = ($env === 'TEST' ? self::API_BASE_URL_TEST : self::API_BASE_URL_PROD);
        return $return . $endpoint;
    }
}
