<?php

namespace Ecomerciar\CruzDelSur\Lib;

use Ecomerciar\CruzDelSur\Lib\RestClient;

class Api
{
    const XML_ENV_YESNO = 'carriers/cruzdelsur/ambiente';
    const XML_ENV_PROD_VALUE = 'carriers/cruzdelsur/urlenvprod';
    const XML_ENV_TEST_VALUE = 'carriers/cruzdelsur/urlenvtest';

    /**
     * @var \Ecomerciar\CruzDelSur\Helper\Helper
     */
    protected $_helper;
    protected $_restclient;
     /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    public function __construct(
        \Ecomerciar\CruzDelSur\Helper\Helper $helper,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        RestClient $restclient
    ) {
        $this->_helper = $helper;
        $this->scopeConfig = $scopeConfig;
        $this->_restclient = $restclient;
    }

    /**
     * Gets a quote from a checkout session
     *
     * @param object $checkoutSession
     * @param Request $request
     * @return array|bool
     */
    public function get_price($magento_quote, $request)
    {
        $shippingAddress = $magento_quote->getShippingAddress();
        $postcode = $shippingAddress->getPostcode();
        if (empty($shippingAddress) || empty($postcode)) return false;
        $products = $this->_helper->getProducts($request->getAllItems());
        if (empty($products)) {
            $this->_helper->log_error(__FUNCTION__ . ' - 1 o más productos con dimensiones/peso en 0');
            return false;
        }
        $total_volume = $total_weight = $total_price = 0;
        foreach ($products as $product) {
            for ($i = 0; $i < $product['quantity']; $i++) {
                if (empty($product['volume'])) {
                    $total_volume += ($product['height'] * $product['length'] * $product['width']);
                } else {
                    $total_volume += $product['volume'];
                }
                $total_weight += $product['weight'];
                $total_price += $product['price'];
            }
        }
        $params = [
            'idcliente' => $this->_helper->get_config('clientid') . '_23',
            'ulogin' => $this->_helper->get_config('username'),
            'uclave' => $this->_helper->get_config('password'),
            'volumen' => $total_volume,
            'peso' => $total_weight,
            'codigopostal' => $postcode,
            'localidad' => $shippingAddress->getCity(),
            'valor' => $total_price,
            'contrareembolso' => 0,
        ];
        $response = $this->_restclient->get("NuevaCotXVol", $params);
        if (!empty($this->_helper->get_config('debug'))) {
            $this->_helper->log_info(__FUNCTION__ . ' - DEBUG cotizando, data enviada: ' . json_encode($params));
            $this->_helper->log_info(__FUNCTION__ . ' - DEBUG cotizando, data recibida: ' . json_encode($response));
        }
        if ($response['status'] === 200) {
            return $this->handle_price_response($response['response']);
        } else {
            $this->_helper->log_error(__FUNCTION__ . ' - Error cotizando, data enviada: ' . json_encode($params));
            $this->_helper->log_error(__FUNCTION__ . ' - Error cotizando, data recibida: ' . json_encode($response));
            return false;
        }
    }

    /**
     * Handles and formats the price response
     *
     * @param array $response
     * @return array
     */
    private function handle_price_response($response)
    {
        $formatted_rates = [];
        $rates = $response['Cotizaciones'];
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $cart = $objectManager->get('\Magento\Checkout\Model\Cart');
        $grandTotal = (float)$cart->getQuote()->getGrandTotal();
        $enviogratis = $this->_helper->get_config('enviogratis');
        $enviogratisdesde = $enviogratis?$this->_helper->get_config('enviogratisdesde'):999999999;

        $formatted_rate = [];
        foreach ($rates as $rate) {
            $this->_helper->log_info(__FUNCTION__ . ' - DEBUG cotizando,rates: ' . json_encode($rates));
            if ($enviogratis && $grandTotal > (float)$enviogratisdesde) {

                if ($this->_helper->get_config('enviogratis')) {
                    $formatted_rate = [
                        'id' => $rate['CodigoLinea'],
                        'name' => 'Envio Gratis',
                        'price' => '0',
                        'from' => $rate['HorasDesde'],
                        'to' => $rate['HorasHasta']
                    ];
                }
            } else {
                $formatted_rate = [
                    'id' => $rate['CodigoLinea'],
                    'name' => (empty($rate['DescripcionLarga']) ? $rate['Descripcion'] : $rate['DescripcionLarga']),
                    'price' => $rate['Valor'],
                    'from' => $rate['HorasDesde'],
                    'to' => $rate['HorasHasta']
                ];
            }
            $formatted_rate['full_name'] = $formatted_rate['name'] . ' (' . $formatted_rate['from'] . '-' . $formatted_rate['to'] . ' Hrs)';
            $formatted_rates[] = $formatted_rate;
        }



        return $formatted_rates;
    }

    /**
     * Process an order to Cruz Del Sur
     *
     * @param object $order
     * @param array $metadata Cruz Del Sur's metadata
     * @return array|bool
     */
    public function process_order($order)
    {
        $metadata = $this->_helper->db_get_meta($order->getId());

        $metadata = unserialize($metadata['meta_value']);



        if (empty($metadata['id'])) return false;

        $shippingAddress = $order->getShippingAddress();
        $postcode = $shippingAddress->getPostcode();
        if (empty($shippingAddress) || empty($postcode)) return false;
        $products = $this->_helper->getProducts($order->getAllItems());
        if (empty($products)) {
            $this->_helper->log_error(__FUNCTION__ . ' - 1 o más productos con dimensiones/peso en 0');
            return false;
        }

        $address = $this->_helper->get_address($shippingAddress);
      //  $address = $shippingAddress->getStreetLine(1);

        /*if (!empty($shippingAddress->getStreetLine(2))) $address .= ', ' . $shippingAddress->getStreetLine(2);
        if (!empty($shippingAddress->getStreetLine(3))) $address .= ', ' . $shippingAddress->getStreetLine(3);
        if (!empty($shippingAddress->getCity())) $address .= ', ' . $shippingAddress->getCity() . ' ' . $postcode;
        if (!empty($shippingAddress->getRegion())) $address .= ', ' . $shippingAddress->getRegion();*/
        $params = [
            'idcliente' => $this->_helper->get_config('clientid') . '_23',
            'idlinea' => $metadata['id'],
            'ulogin' => $this->_helper->get_config('username'),
            'uclave' => $this->_helper->get_config('password'),
            'documento' => '',
            'nombre' => $order->getCustomerName(),
            'telefono' => $shippingAddress->getTelephone(),
            'email' => $order->getCustomerEmail(),
            'domicilio' => $address,
            'bultos' => $this->_helper->get_config('cantidadbultos'),
            'referencia' => $order->getIncrementId()
        ];

        $response = $this->_restclient->get("PedirDespachoCotizacion", $params);

        if (!empty($this->_helper->get_config('debug'))) {
            $this->_helper->log_info(__FUNCTION__ . ' - DEBUG Procesando despacho, data enviada: ' . json_encode($params));
            $this->_helper->log_info(__FUNCTION__ . ' - DEBUG Procesando despacho, data recibida: ' . json_encode($response));
        }
        $responseCode = isset($response['response']['Respuesta'][0]['Estado']) ? $response['response']['Respuesta'][0]['Estado'] : false;
        if ($response['status'] === 200 || $responseCode !== 0) {
            $response = $response['response'];
            return $response['Respuesta'][0]['NIC'];
        } else {
            $this->_helper->log_error(__FUNCTION__ . ' - Error Procesando despacho, data enviada: ' . json_encode($params));
            $this->_helper->log_error(__FUNCTION__ . ' - Error Procesando despacho, data recibida: ' . json_encode($response));
            return false;
        }
    }

    /**
     * Confirms an order
     *
     * @param Order $order
     * @return array|false
     */
    public function confirm_order($order)
    {
        $metadata = $this->_helper->db_get_meta($order->getId());
        $quote_number = $metadata['quote_number'];
        $metadata = unserialize($metadata['meta_value']);
        if (empty($metadata['id'])) return false;
        $holidays = ['01/01', '04/03', '05/03', '24/03', '02/04', '19/04', '01/05', '25/05', '20/06', '09/07', '08/12', '25/12', '31/12'];
        $timezone = new \DateTimeZone('America/Argentina/Buenos_Aires');
        $today = \DateTime::createFromFormat('U', time(), $timezone);
        $today->modify('+1 day');
        if ($today->format('N') > 5 || in_array($today->format('M/D'), $holidays) || date('H') > 18) {
            $i = 0;
            do {
                $today->modify('+1 day');
                $i++;
            } while (($today->format('N') > 5 || in_array($today->format('M/D'), $holidays)));
        }
        $params = [
            'idcliente' => $this->_helper->get_config('clientid') . '_23',
            'ulogin' => $this->_helper->get_config('username'),
            'uclave' => $this->_helper->get_config('password'),
            'idsCotizaciones' => $quote_number,
            'nombre' => $this->_helper->get_config('nombre'),
            'telefono' => $this->_helper->get_config('telefono'),
            'email' => $this->_helper->get_config('email'),
            'domicilio' => $this->_helper->get_config('domicilio'),
            'entreCalles' => (empty($this->_helper->get_config('entrecalles')) ? '' : $this->_helper->get_config('entrecalles')),
            'localidad' => $this->_helper->get_config('localidad'),
            'codigopostal' => $this->_helper->get_config('codigopostal'),
            'observaciones' => (empty($this->_helper->get_config('observaciones')) ? '' : $this->_helper->get_config('observaciones')),
            'fecha' => $today->format('Ymd'),
            'hora1Desde' => $this->_helper->get_config('hora1desde'),
            'hora1Hasta' => $this->_helper->get_config('hora1hasta'),
            'hora2Desde' => ($this->_helper->get_config('hora2desde') === '0' ? '' : $this->_helper->get_config('hora2desde')),
            'hora2Hasta' => ($this->_helper->get_config('hora2hasta') === '0' ? '' : $this->_helper->get_config('hora2hasta'))
        ];
        $response = $this->_restclient->get("GenerarPedidoDeRetiro", $params);
        if (!empty($this->_helper->get_config('debug'))) {
            $this->_helper->log_info(__FUNCTION__ . ' - DEBUG Procesando retiro, data enviada: ' . json_encode($params));
            $this->_helper->log_info(__FUNCTION__ . ' - DEBUG Procesando retiro, data recibida: ' . json_encode($response));
        }
        if ($response['status'] === 200) {
            $response = $response['response'];
            if ($response['Respuesta'][0]['Estado'] === 0) {
                return $response['Respuesta'][0]['PedidoDeRetiro'];
            }
            return false;
        } else {
            $this->_helper->log_error(__FUNCTION__ . ' - Error Procesando retiro, data enviada: ' . json_encode($params));
            $this->_helper->log_error(__FUNCTION__ . ' - Error Procesando retiro, data recibida: ' . json_encode($response));
            return false;
        }
    }

    /**
     * Gets url for a shipping label
     *
     * @param Order $order
     * @param string $type
     * @return string
     */
    public function get_shipping_label_url($order, $type = 'PDF')
    {
        $order_id = $order->getId();
        $metadata = $this->_helper->db_get_meta($order_id);
        $quote_number = $metadata['quote_number'];
        $data = [
            'idcliente' => $this->_helper->get_config('clientid') . '_23',
            'ulogin' => $this->_helper->get_config('username'),
            'uclave' => $this->_helper->get_config('password')
        ];
        if ($type === 'PDF') {
            $data['tamanioHoja'] = $this->_helper->get_config('tamaniohoja');
            $data['posicionArrancar'] = '1';
        } else {
            $data['paraReemplazarCantidades'] = '';
        }
        $url = $this->_restclient->get_url('Etiquetas' . $type . '/' . $quote_number, $data);
        if (!empty($this->_helper->get_config('debug'))) {
            $this->_helper->log_info(__FUNCTION__ . ' - DEBUG Generando url para etiqueta, url generada: ' . $url);
        }
        return $url;
    }

    /**
     * Gets tracking status for a shipment
     *
     * @param string $tracking_number
     * @param string $type
     * @return array|false
     */
    public function get_tracking($tracking_number)
    {
        $data = [
            'idcliente' => $this->_helper->get_config('clientid') . '_23',
            'ulogin' => $this->_helper->get_config('username'),
            'uclave' => $this->_helper->get_config('password'),
            'nic' => $tracking_number
        ];
        $response = $this->_restclient->get('TrackingDeUnNIC/', $data);
        if (!empty($this->_helper->get_config('debug'))) {
            $this->_helper->log_info(__FUNCTION__ . ' - DEBUG Consultando tracking, data enviada: ' . json_encode($data));
            $this->_helper->log_info(__FUNCTION__ . ' - DEBUG Consultando tracking, data recibida: ' . json_encode($response));
        }
        if ($response['status'] === 200) {
            $response = $response['response'];
            if (isset($response['Respuesta'][0]['Estado']) && $response['Respuesta'][0]['Estado'] === 0) {
                return (empty($response['Tracking']) ? 'Pedido sin movimientos' : $response['Tracking']);
            } else {
                return (empty($response['Respuesta'][0]['Descripcion']) ? 'Hubo un error al rastrear el pedido' : $response['Respuesta'][0]['Descripcion']);
            }
        } else {
            $this->_helper->log_error(__FUNCTION__ . ' - Error rastreando pedido ' . $tracking_number . ', data enviada: ' . json_encode($data));
            $this->_helper->log_error(__FUNCTION__ . ' - Error rastreando pedido ' . $tracking_number . ', data recibida: ' . json_encode($response));
            return false;
        }
    }
}
