<?php

namespace Ecomerciar\CruzDelSur\Helper;

class Helper
{
    const CRUZDELSUR_LENGTH_UNIT = 'cm';
    const CRUZDELSUR_WEIGHT_UNIT = 'gr';
    const SHIPPING_CARRIER_CODE = 'cruzdelsur';

    /*
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;
    /*
     * @var \Ecomerciar\CruzDelSur\Helper\DimensionsHelper
     */
    protected $dimensionsHelper;
    /*
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;
    /*
     * @var \Magento\Framework\App\ResourceConnection
     */
    protected $resource;

    /**
     * Constructor
     *
     * @param \Psr\Log\LoggerInterface $logger
     * @param \Ecomerciar\CruzDelSur\Helper\DimensionsHelper $dimensionsHelper
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Framework\App\ResourceConnection $resource
     */
    public function __construct(
        \Psr\Log\LoggerInterface $logger,
        \Ecomerciar\CruzDelSur\Helper\DimensionsHelper $dimensionsHelper,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\App\ResourceConnection $resource
    ) {
        $this->_logger = $logger;
        $this->_dimensionsHelper = $dimensionsHelper;
        $this->_scopeConfig = $scopeConfig;
        $this->_resource = $resource;
    }

    /**
     * Returns a formatted array of products
     *
     * @param $items
     * @return array|bool
     */
    public function getProducts($items)
    {
        $error = false;
        $items = $this->getAllItems($items);
        $dimensions = [];
        foreach ($items as $item) {
            $itemQty = $item->getQty();
            if (empty($itemQty)) $itemQty = $item->getQtyOrdered();
            $dimension = [
                'name' => $item->getName(),
                'weight' => $this->_dimensionsHelper->getShippingDimension($item, 'weight'),
                'quantity' => $itemQty,
                'price' => $item->getPrice(),
                'height' => $this->_dimensionsHelper->getShippingDimension($item, 'height'),
                'length' => $this->_dimensionsHelper->getShippingDimension($item, 'length'),
                'width' => $this->_dimensionsHelper->getShippingDimension($item, 'width'),
                'volume' => $this->_dimensionsHelper->getShippingDimension($item, 'volume')
            ];

            foreach ($dimension as $key => $value) {
                $this->log_error(__FUNCTION__ .$key ."=" .$value);
                if (empty($value) && $key !== 'volume') {
                    $error = true;
                }
            }
            if ($error) {
                break;
            }
            $dimensions[] = $dimension;
        }
        if ($error) {
            return false;
        }
        return $dimensions;
    }

    /**
     * Returns a Formatted address
     *
     * @param ShippingAddress $orderAddress
     * @return array|bool
     */
    public function get_address($orderAddress)
    {
        if (empty($orderAddress)) {
            return false;
        }
        $street_name = $street_number = $floor = $apartment = "";
        $baddressArray = $orderAddress->getStreet();

        if (isset($baddressArray[0])) {
            $street_name = str_replace('"', '', $baddressArray[0]);
        } else {
            $street_name = '';
        }
        $street_number = isset($baddressArray[1]) ? $baddressArray[1] : '';
        $floor = isset($baddressArray[2]) ? $baddressArray[2] : '';
        $apartment = isset($baddressArray[3]) ? $baddressArray[3] : '';
//        if (!empty($shipping_line_2)) {
//            //there is something in the second line. Let's find out what
//            $fl_apt_array = $this->get_floor_and_apt($shipping_line_2);
//            $floor = $fl_apt_array[0];
//            $apartment = $fl_apt_array[1];
//        }
//
//        //Now let's work on the first line
//        preg_match('/(^\d*[\D]*)(\d+)(.*)/i', $shipping_line_1, $res);
//        $line1 = $res;
//
//        if ((isset($line1[1]) && !empty($line1[1]) && $line1[1] !== " ") && !empty($line1)) {
//            //everything's fine. Go ahead
//            if (empty($line1[3]) || $line1[3] === " ") {
//                //the user just wrote the street name and number, as he should
//                $street_name = trim($line1[1]);
//                $street_number = trim($line1[2]);
//                unset($line1[3]);
//            } else {
//                //there is something extra in the first line. We'll save it in case it's important
//                $street_name = trim($line1[1]);
//                $street_number = trim($line1[2]);
//                $shipping_line_2 = trim($line1[3]);
//
//                if (empty($floor) && empty($apartment)) {
//                    //if we don't have either the floor or the apartment, they should be in our new $shipping_line_2
//                    $fl_apt_array = $this->get_floor_and_apt($shipping_line_2);
//                    $floor = $fl_apt_array[0];
//                    $apartment = $fl_apt_array[1];
//                } elseif (empty($apartment)) {
//                    //we've already have the floor. We just need the apartment
//                    $apartment = trim($line1[3]);
//                } else {
//                    //we've got the apartment, so let's just save the floor
//                    $floor = trim($line1[3]);
//                }
//            }
//        } else {
//            //the user didn't write the street number. Maybe it's in the second line
//            //given the fact that there is no street number in the fist line, we'll asume it's just the street name
//            $street_name = $shipping_line_1;
//
//            if (!empty($floor) && !empty($apartment)) {
//                //we are in a pickle. It's a risky move, but we'll move everything one step up
//                $street_number = $floor;
//                $floor = $apartment;
//                $apartment = "";
//            } elseif (!empty($floor) && empty($apartment)) {
//                //it seems the user wrote only the street number in the second line. Let's move it up
//                $street_number = $floor;
//                $floor = "";
//            } elseif (empty($floor) && !empty($apartment)) {
//                //I don't think there's a chance of this even happening, but let's write it to be safe
//                $street_number = $apartment;
//                $apartment = "";
//            }
//        }
        //return array('street' => $street_name, 'number' => $street_number, 'floor' => $floor, 'apartment' => $apartment);
        return $street_name . ' '. $street_number . ' ' .$floor.$apartment;
    }

    /**
     * Gets floor and Apt for an Address string
     *
     * @param string $fl_apt
     * @return array
     */
    public function get_floor_and_apt($fl_apt)
    {
        //firts we'll asume the user did things right. Something like "piso 24, depto. 5h"
        preg_match('/(piso|p|p.) ?(\w+),? ?(departamento|depto|dept|dpto|dpt|dpt.º|depto.|dept.|dpto.|dpt.|apartamento|apto|apt|apto.|apt.) ?(\w+)/i', $fl_apt, $res);
        $line2 = $res;

        if (!empty($line2)) {
            //everything was written great. Now lets grab what matters
            $floor = trim($line2[2]);
            $apartment = trim($line2[4]);
        } else {
            //maybe the user wrote something like "depto. 5, piso 24". Let's try that
            preg_match('/(departamento|depto|dept|dpto|dpt|dpt.º|depto.|dept.|dpto.|dpt.|apartamento|apto|apt|apto.|apt.) ?(\w+),? ?(piso|p|p.) ?(\w+)/i', $fl_apt, $res);
            $line2 = $res;
        }

        if (!empty($line2) && empty($apartment) && empty($floor)) {
            //apparently, that was the case. Guess some people just like to make things difficult
            $floor = trim($line2[4]);
            $apartment = trim($line2[2]);
        } else {
            //something is wrong. Let's be more specific. First we'll try with only the floor
            preg_match('/^(piso|p|p.) ?(\w+)$/i', $fl_apt, $res);
            $line2 = $res;
        }

        if (!empty($line2) && empty($floor)) {
            //now we've got it! The user just wrote the floor number. Now lets grab what matters
            $floor = trim($line2[2]);
        } else {
            //still no. Now we'll try with the apartment
            preg_match('/^(departamento|depto|dept|dpto|dpt|dpt.º|depto.|dept.|dpto.|dpt.|apartamento|apto|apt|apto.|apt.) ?(\w+)$/i', $fl_apt, $res);
            $line2 = $res;
        }

        if (!empty($line2) && empty($apartment) && empty($floor)) {
            //success! The user just wrote the apartment information. No clue why, but who am I to judge
            $apartment = trim($line2[2]);
        } else {
            //ok, weird. Now we'll try a more generic approach just in case the user missplelled something
            preg_match('/(\d+),? [a-zA-Z.,!*]* ?([a-zA-Z0-9 ]+)/i', $fl_apt, $res);
            $line2 = $res;
        }

        if (!empty($line2) && empty($floor) && empty($apartment)) {
            //finally! The user just missplelled something. It happens to the best of us
            $floor = trim($line2[1]);
            $apartment = trim($line2[2]);
        } else {
            //last try! This one is in case the user wrote the floor and apartment together ("12C")
            preg_match('/(\d+)(\D*)/i', $fl_apt, $res);
            $line2 = $res;
        }

        if (!empty($line2) && empty($floor) && empty($apartment)) {
            //ok, we've got it. I was starting to panic
            $floor = trim($line2[1]);
            $apartment = trim($line2[2]);
        } elseif (empty($floor) && empty($apartment)) {
            //I give up. I can't make sense of it. We'll save it in case it's something useful
            $floor = $fl_apt;
        }

        return array($floor, $apartment);
    }

    /**
     * Return items for further shipment rate evaluation. We need to pass children of a bundle instead passing the
     * bundle itself, otherwise we may not get a rate at all (e.g. when total weight of a bundle exceeds max weight
     * despite each item by itself is not)
     *
     * @return array
     */
    private function getAllItems($allItems)
    {
        $items = [];
        foreach ($allItems as $item) {
            /* @var $item Mage_Sales_Model_Quote_Item */
            if ($item->getProduct()->isVirtual() || $item->getParentItem()) {
                // Don't process children here - we will process (or already have processed) them below
                continue;
            }

            if ($item->getHasChildren() && $item->isShipSeparately()) {
                foreach ($item->getChildren() as $child) {
                    if (!$child->getFreeShipping() && !$child->getProduct()->isVirtual()) {
                        $items[] = $child;
                    }
                }
            } else {
                // Ship together - count compound item as one solid
                $items[] = $item;
            }
        }

        return $items;
    }

    /**
     * Gets a config in the settings
     *
     * @param string $key
     * @return string
     */
    public function get_config($key)
    {
        return $this->_scopeConfig->getValue('carriers/cruzdelsur/' . $key);
    }

    /**
     * Gets the table name for our shipping method
     *
     * @param string $tablename
     * @return string
     */
    public function get_table_db($tablename = 'cruzdelsur_orders')
    {
        return $this->_resource->getTableName($tablename);
    }

    /**
     * Gets a database connection
     *
     * @return \Magento\Framework\DB\Adapter\AdapterInterface
     */
    public function get_connection_db()
    {
        return $this->_resource->getConnection();
    }

    /**
     * Inserts metadata in the shipping method's table
     *
     * @param array $data
     * @return integer|false
     */
    public function db_insert_meta($data)
    {
        $connection = $this->get_connection_db();
        return $connection->insert($this->get_table_db(), $data);
    }

    /**
     * Updates metadata in the shipping method's table
     *
     * @param [type] $data
     * @param [type] $where
     * @return integer|false
     */
    public function db_update_meta($data, $where)
    {
        $connection = $this->get_connection_db();
        return $connection->update($this->get_table_db(), $data,$where);
        //return $connection->update($this->get_table_db(), $data)->where('order_id' . ' = ?', $where);
    }

    /**
     * Gets metadata in the shipping method's table
     *
     * @param string $value
     * @param string $where
     * @return array|false
     */
    public function db_get_meta($value, $where = 'order_id', $statementToGet = 'row', $codition = ' = ?')
    {
        $connection = $this->get_connection_db();
        $select = $connection->select()
            ->from($this->get_table_db())
            ->where($where . $codition, $value);
        $res = [];
        if($statementToGet == 'row') {
            $res = $connection->fetchRow($select);
        }
        elseif($statementToGet == 'all'){
            $res = $connection->fetchAll($select);
        }
        return $res;
    }

    /**
     * Logs an info message
     *
     * @param string $msg
     * @param string $type
     * @return void
     */
    public function log_info($msg, $type = 'array')
    {
        if (is_array($msg) && $type = 'array') {
            $msg = print_r($msg, true);
        } else if (is_array($msg) && $type = 'json') {
            $msg = json_encode($msg, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        }
        $this->_logger->info($msg);
    }

    /**
     * Logs an error message
     *
     * @param string $msg
     * @param string $type
     * @return void
     */
    public function log_error($msg, $type = 'array')
    {
        if (is_array($msg) && $type = 'array') {
            $msg = print_r($msg, true);
        } else if (is_array($msg) && $type = 'json') {
            $msg = json_encode($msg, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        }
        $this->_logger->error($msg);
    }

    /**
     * Logs a debug message
     *
     * @param string $msg
     * @param string $type
     * @return void
     */
    public function log_debug($msg, $type = 'array')
    {
        if (is_array($msg) && $type = 'array') {
            $msg = print_r($msg, true);
        } else if (is_array($msg) && $type = 'json') {
            $msg = json_encode($msg, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        }
        $this->_logger->debug($msg);
    }
}
